/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef UUID_f014b81a_01d2_11e4_98ee_7befe8cb5334
#define UUID_f014b81a_01d2_11e4_98ee_7befe8cb5334

/**
 * \file vncwifidisplaytypes.h
 *
 * This file defines types used by the VNC WiFi Display SDK. You should
 * normally include vncwifidisplaysdk.h instead of including this file
 * directly.
 */

#include "vncint.h"
#include "vnccall.h"
#include "vnccommontypes.h"
#include "vncmirrorlinktypes.h"

/**
 * \file keysymdef.h
 *
 * This file defines constants representing key symbols as they are transmitted
 * to MirrorLink devices. You should normally include vncwifidisplaysdk.h
 * instead of including keysymdef.h directly.
 *
 * keysymdef.h is the copyright of The Open Group. For full copyright
 * information, please see the copyright notice in the file itself.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkKeyEvent
 */
#ifndef XK_VoidSymbol
#include "keysymdef.h"
#endif

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \typedef VNCWiFiDisplaySink
 * \brief Represents a single instance of a VNC WiFi Display Sink.
 *
 * To create a VNC WiFi Display Sink, call VNCWiFiDisplaySinkCreate().
 *
 * This type is opaque; use the VNC WiFi Display SDK APIs to interact with it.
 *
 * \see VNCWiFiDisplaySinkCreate
 */
typedef struct VNCWiFiDisplaySinkImplBase VNCWiFiDisplaySink;

/**
 * \brief Error codes that may be notified via the
 * VNCWiFiDisplaySinkErrorCallback().
 *
 * Any error indicates that the VNC WiFi Display Sink has stopped, and that the
 * WiFi Display sink thread is about to exit.
 *
 * \see VNCWiFiDisplaySinkErrorCallback
 */
typedef enum
{
  /**
   * \brief Placeholder value for variable initialization.
   */
  VNCWiFiDisplayErrorNone = 0,

  /**
   * \brief A parameter supplied to an SDK API method is not valid.
   */
  VNCWiFiDisplayErrorInvalidParameter = 1,

  /**
   * \brief The requested action cannot be performed in the VNC WiFi Display
   * Sink's current state.
   *
   * \see VNCWiFiDisplaySinkStateChangedCallback, VNCWiFiDisplayState
   */
  VNCWiFiDisplayErrorInvalidState = 2,

  /**
   * \brief The application called an SDK API that may not be called while the
   * WiFi Display sink thread is running.
   */
  VNCWiFiDisplayErrorIllegalWhileRunning = 3,

  /**
   * \brief The application called an SDK API that may not be called unless the
   * WiFi Display sink thread is running.
   */
  VNCWiFiDisplayErrorIllegalWhileNotRunning = 4,

  /**
   * \brief The request failed because the network port in question is already
   * in use.
   */
  VNCWiFiDisplayErrorPortInUse = 5,

  /**
   * \brief Permission to access the specified resource or entity was denied by
   * the operating system.
   */
  VNCWiFiDisplayErrorPermissionDenied = 6,

  /**
   * \brief VNCWiFiDisplaySinkStop() was called (either explicitly or from
   * VNCWiFiDisplaySinkDestroy()).
   */
  VNCWiFiDisplayErrorStopped = 7,

  /**
   * \brief The operation failed for an unknown reason.
   */
  VNCWiFiDisplayErrorFailed = 8,

  /**
   * \brief The required object, entity or data was not found.
   */
  VNCWiFiDisplayErrorNotFound = 9,

  /**
   * \brief The operation failed because the thing being added or registered
   * already exists.
   */
  VNCWiFiDisplayErrorAlreadyExists = 10,

  /**
   * \brief The operation failed because it is not supported.
   */
  VNCWiFiDisplayErrorNotSupported = 11,

  /**
   * \brief This API function is not implemented for the current platform.
   */
  VNCWiFiDisplayErrorNotImplemented = 12,

  /**
   * \brief A license could not be added because it is invalid.
   */
  VNCWiFiDisplayErrorLicenseNotValid = 13,

  /**
   * \brief A feature of the SDK cannot be used because it is not licensed.
   */
  VNCWiFiDisplayErrorFeatureNotLicensed = 14,

  /**
   * \brief An audio mode which is mandatory in this context has not been
   * specified.
   */
  VNCWiFiDisplayErrorMissingAudioMode = 15,

  /**
   * \brief It was not possible to load the VNC WiFi Display Decoder Plugin.
   */
  VNCWiFiDisplayErrorDecoderPluginLoadFailed = 16,

  /**
   * \brief A decoder plugin has not been registered, or does not support the
   * requested audio or H.264 codec.
   */
  VNCWiFiDisplayErrorMissingDecoderPlugin = 17,

  /**
   * \brief The request failed because no native resolution has been set.
   */
  VNCWiFiDisplayErrorNoNativeResolution = 18,

  /**
   * \brief The request failed because no active connector type has been set.
   */
  VNCWiFiDisplayErrorNoActiveConnectorType = 19,

  /**
   * \brief The request failed because no WiFi Display Source has been set.
   */
  VNCWiFiDisplayErrorNoSource = 20,

  /**
   * \brief The request failed because a mandatory audio codec has not been
   * specified.
   */
  VNCWiFiDisplayErrorMissingAudioCodec = 21,

  /**
   * \brief The request failed because a mandatory H.264 codec has not been
   * specified.
   */
  VNCWiFiDisplayErrorMissingH264Codec = 22,

  /**
   * \brief The requested action cannot be performed in the VNC WiFi Display
   * Sink's current UIBC state.
   *
   * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
   */
  VNCWiFiDisplayErrorInvalidUIBCState = 23,

  /**
   * \brief The request failed because it refers to an unknown MirrorLink UI or
   * audio context event.
   */
  VNCWiFiDisplayErrorUnknownMirrorLinkContextEvent = 24,

  /**
   * \brief The WiFi Display peer network is unreachable.
   */
  VNCWiFiDisplayErrorNetworkUnreachable = 25,

  /**
   * \brief The WiFi Display peer host is unreachable.
   */
  VNCWiFiDisplayErrorHostUnreachable = 26,

  /**
   * \brief A connection to a WiFi Display peer service timed out.
   */
  VNCWiFiDisplayErrorConnectionTimedOut = 27,

  /**
   * \brief A connection to a WiFi Display peer service was refused.
   */
  VNCWiFiDisplayErrorConnectionRefused = 28,

  /**
   * \brief A connection to a WiFi Display peer service was lost.
   */
  VNCWiFiDisplayErrorDisconnected = 29,

  /**
   * \brief A connection to a WiFi Display peer service failed because an
   * invalid port was specified.
   */
  VNCWiFiDisplayErrorBadPort = 30,

  /**
   * \brief A WiFi Display decoder is already loaded.
   */
  VNCWiFiDisplayErrorDecoderAlreadyLoaded = 31,

  /**
   * \brief It was not possible to create an instance of the WiFi Display
   * decoder.
   */
  VNCWiFiDisplayErrorDecoderCreateFailed = 32,

  /**
   * \brief The RTSP session was aborted, due to an error or timeout.
   */
  VNCWiFiDisplayErrorRTSPSessionAborted = 33,

} VNCWiFiDisplayError;

/**
 * \brief The type of WiFi Display sink.
 *
 * Currently, this SDK only allows VNC WiFi Display Sink instances to act as
 * primary sinks.
 */
typedef enum
{
  /**
   * \brief A primary WiFi Display sink.
   */
  VNCWiFiDisplaySinkTypePrimary = 0,

} VNCWiFiDisplaySinkType;

/**
 * \brief Properties that may be queried via VNCWiFiDisplaySinkGetProperty() or
 * VNCWiFiDisplaySinkGetPropertyString().
 *
 * Integer-valued properties must be queried with
 * VNCWiFiDisplaySinkGetProperty(). Passing the identifier for an
 * integer-valued property to VNCWiFiDisplaySinkGetPropertyString() will result
 * in a NULL return.
 *
 * String-valued properties must be queried with
 * VNCWiFiDisplaySinkGetPropertyString(). Passing the identifier for a
 * string-valued property to VNCWiFiDisplaySinkGetProperty() will result in a
 * zero return.
 */
typedef enum
{
  /**
   * \brief Whether or not the WiFi Display Sink thread is running (integer).
   *
   * The value is non-zero if the VNC WiFi Display Sink has been started, and
   * not yet stopped. Otherwise, the value is zero.
   */
  VNCWiFiDisplayPropertyIsRunning = 0,

} VNCWiFiDisplayProperty;

/**
 * \brief Resolution types supported in WiFi Display.
 *
 * Refer to Table 3-4 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */
typedef enum
{
  /**
   * \brief A Consumer Electronics Association (CEA) resolution.
   *
   * \see ResolutionCEA
   */
  VNCWiFiDisplayResolutionTypeCEA = 0x00,

  /**
   * \brief A Video Electronics Standard Assocation (VESA) resolution.
   *
   * \see ResolutionVESA
   */
  VNCWiFiDisplayResolutionTypeVESA = 0x01,

  /**
   * \brief A HandHeld (HH) resolution.
   *
   * \see ResolutionHH
   */
  VNCWiFiDisplayResolutionTypeHH = 0x02,

} VNCWiFiDisplayResolutionType;


/**
 * \defgroup ResolutionCEA CEA resolutions and refresh rates.
 *
 * This group defines the bitmap values that represent the supported CEA
 * resolutions and refresh rates.
 *
 * Refer to Table 5-10 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 *
 * \see VNCWiFiDisplayResolutionTypeCEA.
 */

/**
 * \ingroup ResolutionCEA
 * 640x480 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_640_480_p60     0x00000001

/**
 * \ingroup ResolutionCEA
 * 720x480 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_720_480_p60     0x00000002

/**
 * \ingroup ResolutionCEA
 * 720x480 pixels, interlaced, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_720_480_i60     0x00000004

/**
 * \ingroup ResolutionCEA
 * 720x576 pixels, progressive, 50Hz.
 */
#define VNCWiFiDisplayResolutionCEA_720_576_p50     0x00000008

/**
 * \ingroup ResolutionCEA
 * 720x576 pixels, interlaced, 50Hz.
 */
#define VNCWiFiDisplayResolutionCEA_720_576_i50     0x00000010

/**
 * \ingroup ResolutionCEA
 * 1280x720 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1280_720_p30    0x00000020

/**
 * \ingroup ResolutionCEA
 * 1280x720 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1280_720_p60    0x00000040

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_p30   0x00000080

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_p60   0x00000100

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, interlaced, 60Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_i60   0x00000200

/**
 * \ingroup ResolutionCEA
 * 1280x720 pixels, progressive, 25Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1280_720_p25    0x00000400

/**
 * \ingroup ResolutionCEA
 * 1280x720 pixels, progressive, 50Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1280_720_p50    0x00000800

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, progressive, 25Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_p25   0x00001000

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, progressive, 50Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_p50   0x00002000

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, interlaced, 50Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_i50   0x00004000

/**
 * \ingroup ResolutionCEA
 * 1280x720 pixels, progressive, 24Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1280_720_p24    0x00008000

/**
 * \ingroup ResolutionCEA
 * 1920x1080 pixels, progressive, 24Hz.
 */
#define VNCWiFiDisplayResolutionCEA_1920_1080_p24   0x00010000


/**
 * \defgroup ResolutionVESA VESA resolutions and refresh rates.
 *
 * This group defines the bitmap values that represent the supported VESA
 * resolutions and refresh rates.
 *
 * Refer to Table 5-11 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 *
 * \see VNCWiFiDisplayResolutionTypeVESA.
 */

/**
 * \ingroup ResolutionVESA
 * 800x600 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_800_600_p30    0x00000001

/**
 * \ingroup ResolutionVESA
 * 800x600 pixels, progresive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_800_600_p60    0x00000002

/**
 * \ingroup ResolutionVESA
 * 1024x768 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1024_768_p30   0x00000004

/**
 * \ingroup ResolutionVESA
 * 1024x768 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1024_768_p60   0x00000008

/**
 * \ingroup ResolutionVESA
 * 1152x864 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1152_864_p30   0x00000010

/**
 * \ingroup ResolutionVESA
 * 1152x864 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1152_864_p60   0x00000020

/**
 * \ingroup ResolutionVESA
 * 1280x768 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_768_p30   0x00000040

/**
 * \ingroup ResolutionVESA
 * 1280x768 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_768_p60   0x00000080

/**
 * \ingroup ResolutionVESA
 * 1280x800 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_800_p30   0x00000100

/**
 * \ingroup ResolutionVESA
 * 1280x800 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_800_p60   0x00000200

/**
 * \ingroup ResolutionVESA
 * 1360x768 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1360_768_p30   0x00000400

/**
 * \ingroup ResolutionVESA
 * 1360x768 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1360_768_p60   0x00000800

/**
 * \ingroup ResolutionVESA
 * 1366x768 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1366_768_p30   0x00001000

/**
 * \ingroup ResolutionVESA
 * 1366x768 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1366_768_p60   0x00002000

/**
 * \ingroup ResolutionVESA
 * 1280x1024 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_1024_p30  0x00004000

/**
 * \ingroup ResolutionVESA
 * 1280x1024 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1280_1024_p60  0x00008000

/**
 * \ingroup ResolutionVESA
 * 1400x1050 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1400_1050_p30  0x00010000

/**
 * \ingroup ResolutionVESA
 * 1400x1050 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1400_1050_p60  0x00020000

/**
 * \ingroup ResolutionVESA
 * 1400x900 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1400_900_p30   0x00040000

/**
 * \ingroup ResolutionVESA
 * 1400x900 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1400_900_p60   0x00080000

/**
 * \ingroup ResolutionVESA
 * 1600x900 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1600_900_p30   0x00100000

/**
 * \ingroup ResolutionVESA
 * 1600x900 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1600_900_p60   0x00200000

/**
 * \ingroup ResolutionVESA
 * 1600x1200 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1600_1200_p30  0x00400000

/**
 * \ingroup ResolutionVESA
 * 1600x1200 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1600_1200_p60  0x00800000

/**
 * \ingroup ResolutionVESA
 * 1680x1024 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1680_1024_p30  0x01000000

/**
 * \ingroup ResolutionVESA
 * 1680x1024 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1680_1024_p60  0x02000000

/**
 * \ingroup ResolutionVESA
 * 1680x1050 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1680_1050_p30  0x04000000

/**
 * \ingroup ResolutionVESA
 * 1680x1050 pixels, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1680_1050_p60  0x08000000

/**
 * \ingroup ResolutionVESA
 * 1920x1200 pixels, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionVESA_1920_1200_p30  0x10000000


/**
 * \defgroup ResolutionHH Handheld device resolutions and refresh rates.
 *
 * This group defines the bitmap values that represent resolutions and refresh
 * rates commonly supported by handheld devices.
 *
 * Refer to Table 5-12 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 *
 * \see VNCWiFiDisplayResolutionTypeHH
 */

/**
 * \ingroup ResolutionHH
 * 800x480, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_800_480_p30      0x00000001

/**
 * \ingroup ResolutionHH
 * 800x480, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_800_480_p60      0x00000002

/**
 * \ingroup ResolutionHH
 * 854x480, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_854_480_p30      0x00000004

/**
 * \ingroup ResolutionHH
 * 854x480, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_854_480_p60      0x00000008

/**
 * \ingroup ResolutionHH
 * 864x480, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_864_480_p30      0x00000010

/**
 * \ingroup ResolutionHH
 * 864x480, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_864_480_p60      0x00000020

/**
 * \ingroup ResolutionHH
 * 640x360, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_640_360_p30      0x00000040

/**
 * \ingroup ResolutionHH
 * 640x360, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_640_360_p60      0x00000080

/**
 * \ingroup ResolutionHH
 * 960x540, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_960_540_p30      0x00000100

/**
 * \ingroup ResolutionHH
 * 960x540, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_960_540_p60      0x00000200

/**
 * \ingroup ResolutionHH
 * 848x480, progressive, 30Hz.
 */
#define VNCWiFiDisplayResolutionHH_848_480_p30      0x00000400

/**
 * \ingroup ResolutionHH
 * 848x480, progressive, 60Hz.
 */
#define VNCWiFiDisplayResolutionHH_848_480_p60      0x00000800


/**
 * \brief Supported display connector types.
 *
 * Refer to Table 6-6 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */
typedef enum
{
  /**
   * \brief Video Graphics Array (VGA) connector type.
   */
  VNCWiFiDisplayConnectorTypeVGA = 0,

  /**
   * \brief S-Video connector type.
   */
  VNCWiFiDisplayConnectorTypeSVideo = 1,

  /**
   * \brief Composite video connector type.
   */
  VNCWiFiDisplayConnectorTypeComposite= 2,

  /**
   * \brief Component video connector type.
   */
  VNCWiFiDisplayConnectorTypeComponent = 3,

  /**
   * \brief Digital Video Interface (DVI) connector type.
   */
  VNCWiFiDisplayConnectorTypeDVI = 4,

  /**
   * \brief High-Definition Multimedia Interface (HDMI) connector type.
   */
  VNCWiFiDisplayConnectorTypeHDMI = 5,

  /**
   * \brief Wi-Fi Display connector type.
   */
  VNCWiFiDisplayConnectorTypeWFD = 7,

  /**
   * \brief Japanese D connector type (conforming to the EIAJ RC-5327
   * standard).
   */
  VNCWiFiDisplayConnectorTypeJPD = 8,

  /**
   * \brief Serial Digital Image (SDI) connector type.
   */
  VNCWiFiDisplayConnectorTypeSDI = 9,

  /**
   * \brief Display Port (DP) connector type.
   */
  VNCWiFiDisplayConnectorTypeDP = 10,

  /**
   * \brief Unified Display Interface (UDI) connector type.
   */
  VNCWiFiDisplayConnectorTypeUDI = 12,

  /**
   * \brief A connector type that is not currently covered by the above
   * values.
   */
  VNCWiFiDisplayConnectorTypeOther = 255,

} VNCWiFiDisplayConnectorType;

/**
 * \brief Audio formats supported by WiFi Display sources and sinks.
 *
 * Refer to Table 3-2 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */
typedef enum
{
  /**
   * \brief Linear Pulse Coded Modulation (LPCM) audio format.
   */
  VNCWiFiDisplayAudioFormatLPCM = 0,

  /**
   * \brief Advanced Audio coding (AAC) audio format.
   */
  VNCWiFiDisplayAudioFormatAAC = 1,

  /**
   * \brief Digital Audio Compression Standard (AC-3) audio format.
   */
  VNCWiFiDisplayAudioFormatAC3 = 2,

} VNCWiFiDisplayAudioFormat;


/**
 * \defgroup AudioModeLPCM LPCM audio modes.
 *
 * This group defines the bitmap values that represent the supported LPCM audio
 * modes.
 *
 * Refer to Table 5-21 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */

/**
 * \ingroup AudioModeLPCM
 * 44100 samples per second, 16 bits per channel, 2 channels.
 */
#define VNCWiFiDisplayAudioModeLPCM_44100_16_2      0x00000001

/**
 * \ingroup AudioModeLPCM
 * 48000 samples per second, 16 bits per channel, 2 channels.
 */
#define VNCWiFiDisplayAudioModeLPCM_48000_16_2      0x00000002


/**
 * \defgroup AudioModeAAC AAC audio modes.
 *
 * This group defines the bitmap values that represent the supported AAC audio
 * modes.
 *
 * Refer to Table 5-22 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */

/**
 * \ingroup AudioModeAAC
 * 48000 samples per second, 16 bits per channel, 2 channels.
 */
#define VNCWiFiDisplayAudioModeAAC_48000_16_2       0x00000001

/**
 * \ingroup AudioModeAAC
 * 48000 samples per second, 16 bits per channel, 4 channels.
 */
#define VNCWiFiDisplayAudioModeAAC_48000_16_4       0x00000002

/**
 * \ingroup AudioModeAAC
 * 48000 samples per second, 16 bits per channel, 6 channels.
 */
#define VNCWiFiDisplayAudioModeAAC_48000_16_6       0x00000004

/**
 * \ingroup AudioModeAAC
 * 48000 samples per second, 16 bits per channel, 8 channels.
 */
#define VNCWiFiDisplayAudioModeAAC_48000_16_8       0x00000008


/**
 * \defgroup AudioModeAC3 AC3 audio modes.
 *
 * This group defines the bitmap values that represent the supported AC3 audio
 * modes.
 *
 * Refer to Table 5-23 of the "Wi-Fi Display Technical Specification" published
 * by "Wi-Fi Alliance" at http://www.wi-fi.org for more information.
 */

/**
 * \ingroup AudioModeAC3
 * 48000 samples per second, 16 bits per channel, 2 channels.
 */
#define VNCWiFiDisplayAudioModeAC3_48000_16_2       0x00000001

/**
 * \ingroup AudioModeAC3
 * 48000 samples per second, 16 bits per channel, 4 channels.
 */
#define VNCWiFiDisplayAudioModeAC3_48000_16_4       0x00000002

/**
 * \ingroup AudioModeAC3
 * 48000 samples per second, 16 bits per channel, 6 channels.
 */
#define VNCWiFiDisplayAudioModeAC3_48000_16_6       0x00000004


/**
 * \brief An audio codec supported by a VNC WiFi Display Sink.
 *
 * \see VNCWiFiDisplaySinkAddAudioCodec, VNCWiFiDisplaySinkClearAudioCodecs
 */
typedef struct
{
  /**
   * \brief The audio format supported by this codec.
   */
  VNCWiFiDisplayAudioFormat format;

  /**
   * \brief The audio modes supported by this codec.
   *
   * This is a bitmap of one or more of the values from the appropriate bitmap
   * value group for the VNCWiFiDisplayAudioFormat. At least one audio mode
   * must be supported.
   *
   * \see AudioModeLPCM, AudioModeAAC, AudioModeAC3
   */
  vnc_uint32_t modes;

  /**
   * \brief The best-effort estimate of the audio latency for this codec.
   *
   * This is expressed as a multiple of 5ms units. For example, setting this
   * field to 5 will indicate a latency of 25ms. This may be set to zero if an
   * audio latency estimate cannot be provided.
   */
  vnc_uint8_t latency;

} VNCWiFiDisplayAudioCodec;

/**
 * \brief H.264 profiles supported by a VNC WiFi Display Sink.
 */
typedef enum
{
  /**
   * \brief Constrained Baseline Profile (CBP).
   */
  VNCWiFiDisplayH264ProfileCBP = 0,

  /**
   * \brief Constrained High Profile (CHP).
   */
  VNCWiFiDisplayH264ProfileCHP = 1,

} VNCWiFiDisplayH264Profile;

/**
 * \brief H.264 levels supported by a VNC WiFi Display Sink.
 */
typedef enum
{
  /**
   * \brief H.264 Level 3.1.
   */
  VNCWiFiDisplayH264Level_3_1 = 0,

  /**
   * \brief H.264 Level 3.2.
   */
  VNCWiFiDisplayH264Level_3_2 = 1,

  /**
   * \brief H.264 Level 4.
   */
  VNCWiFiDisplayH264Level_4 = 2,

  /**
   * \brief H.264 Level 4.1.
   */
  VNCWiFiDisplayH264Level_4_1 = 3,

  /**
   * \brief H.264 Level 4.2.
   */
  VNCWiFiDisplayH264Level_4_2 = 4,

} VNCWiFiDisplayH264Level;

/**
 * \brief A H.264 video codec supported by a VNC WiFi Display Sink.
 *
 * \see VNCWiFiDisplaySinkAddH264Codec, VNCWiFiDisplaySinkClearH264Codecs
 */
typedef struct
{
  /**
   * \brief The H.264 profile supported by this codec.
   */
  VNCWiFiDisplayH264Profile profile;

  /**
   * \brief The maximum H.264 level supported by this codec.
   */
  VNCWiFiDisplayH264Level level;

  /**
   * \brief The bitmap of CEA resolutions supported by this codec.
   *
   * This may be zero if no CEA resolutions are supported. Otherwise, this must
   * hold a combination of values from the ResolutionCEA bitmap value group
   * describing the CEA resolutions supported.
   *
   * \see ResolutionCEA
   */
  vnc_uint32_t ceaSupportBitmap;

  /**
   * \brief The bitmap of VESA resolutions supported by this codec.
   *
   * This may be zero if no VESA resolutions are supported. Otherwise, this
   * must hold a combination of values from the ResolutionVESA bitmap values
   * group describing the VESA resolutions supported.
   *
   * \see ResolutionVESA
   */
  vnc_uint32_t vesaSupportBitmap;

  /**
   * \brief The bitmap of common handheld resolutions supported by this codec.
   *
   * This may be zero if no handheld resolutions are supported. Otherwise, this
   * must hold a combination of values from the ResolutionHH bitmap values
   * group describing the handheld resolutions supported.
   *
   * \see ResolutionHH
   */
  vnc_uint32_t hhSupportBitmap;

  /**
   * \brief The best-effort estimate of the video latency for this codec.
   *
   * This is expressed as a multiple of 5ms units. For example, setting this
   * field to 5 will indicate a latency of 25ms. This may be set to zero if a
   * video latency estimate cannot be provided.
   */
  vnc_uint8_t latency;

  /**
   * \brief The minimum number of macroblocks that this codec requires a slice
   * to have.
   *
   * This may be zero if this codec does not support slice encoding.
   */
  vnc_uint16_t minSliceSize;

  /**
   * \brief The maximum number of slices per picture that this codec supports.
   *
   * The SDK will adjust this value to be zero-based. This must be zero if
   * minSliceSize is zero, as this codec does not support slice encoding.
   */
  vnc_uint16_t maxNumSlices;

  /**
   * \brief The ratio of the maximum number of macroblocks to use for a slice
   * to the minimum number of macroblocks.
   *
   * The maximum value that may be specified is 7. This must be zero if
   * minSliceSize is zero, as this codec does not support slice encoding.
   */
  vnc_uint8_t maxSliceSizeRatio;

  /**
   * \brief Whether or not this codec supports video frame skipping.
   *
   * If supported, then a WiFi Display source may choose to skip video frames,
   * depending on the current condition of the WiFi Display session connection.
   */
  vnc_bool_t frameSkippingSupported;

  /**
   * \brief The maximum time-interval between two video frames, when frame
   * skipping is supported.
   *
   * If this codec supports frame skipping, then this specifies the maximum
   * interval between two frames, after skips, in units of 0.5 seconds. For
   * example, a value of 2 indicates an interval of 1 second. A value of zero
   * indicates no maximum interval.
   *
   * The maximum value that may be specified is 7, which represents an interval
   * of 3.5 seconds.
   *
   * This should be zero if frame skipping is not supported.
   */
  vnc_uint8_t frameSkippingMaxInterval;

  /**
   * \brief Whether or not this codec supports dynamic video frame rate
   * changes (without user intervention).
   *
   * If supported, then a WiFi Display source may choose to change the video
   * frame rate during a WiFi Display Capability Re-negotiation phase.
   * Otherwise, video frame rate shall only be allowed to change following user
   * intervention.
   */
  vnc_bool_t frameRateChangeSupported;

} VNCWiFiDisplayH264Codec;

/**
 * \brief A MirrorLink configuration for a User Input Back Channel (UIBC)
 * session.
 */
typedef struct
{
  /**
   * \brief Display resolution width, in pixels.
   */
  vnc_uint16_t displayWidthPixels;

  /**
   * \brief Display resolution height, in pixels.
   */
  vnc_uint16_t displayHeightPixels;

  /**
   * \brief Display width, in millimeters.
   */
  vnc_uint16_t displayWidthMM;

  /**
   * \brief Display height, in millimeters.
   */
  vnc_uint16_t displayHeightMM;

  /**
   * \brief Display distance to user, in millimeters.
   */
  vnc_uint16_t displayDistanceMM;

  /**
   * \brief Bitmask indicating the support for knob key input.
   *
   * The value is a combination of the values in the \ref VNCKnobKeySupport
   * enumeration. For example, if the VNCKnobKeySupport0PushZ bit is set, then
   * the server will respond to key events of type XK_Knob_2D_shift_push(0).
   */
  vnc_uint32_t knobKeySupport;

  /**
   * \brief Bitmask indicating the support for MirrorLink device key input.
   *
   * The value is a combination of the values in the \ref VNCDeviceKeySupport
   * enumeration. For example, if the VNCDeviceKeySupportSoftLeft bit is set,
   * then the server will respond to key events of type XK_Device_Soft_left.
   */
  vnc_uint32_t deviceKeySupport;

  /**
   * \brief Bitmask indicating the support for multimedia key input.
   *
   * The value is a combination of the values in the
   * \ref VNCMultimediaKeySupport enumeration. For example, if the
   * VNCMultimediaKeySupportPlay bit is set, then the server will respond to
   * key events of type XK_Multimedia_Play.
   */
  vnc_uint32_t multimediaKeySupport;

  /**
   * \brief The number of additional function keys supported.
   */
  vnc_uint8_t functionKeyCount;

  /**
   * \brief Whether or not ITU key input is supported.
   */
  vnc_bool_t ituKeysSupported;

  /**
   * \brief Bitmask indicating the support for pointer inputs.
   *
   * The value is a combination of the values in the
   * \ref VNCPointerDeviceButton enumeration. For example, if the
   * VNCPointerDeviceButton1 bit is set, then the server will respond to
   * pointer events with the VNCPointerDeviceButtonLeft bit set.
   */
  vnc_uint8_t pointerButtonMask;

  /**
   * \brief The number of simultaneous touch events supported, minus one.
   * 
   * This must be set to zero if touch events are not supported, along with
   * \ref touchPressureMask.
   */
  vnc_uint8_t touchCount;

  /**
   * \brief The number of distinct touch pressure levels that can be
   * distinguished between.
   *
   * This must be set to zero if touch events are not supported. If touch
   * events are supported, then this may not be zero.
   */
  vnc_uint8_t touchPressureMask;

  /**
   * \brief The maximum supported length of a textual meta information string.
   */
  vnc_uint16_t metaTextMaxLength;

  /**
   * \brief Whether or not cut text events are supported.
   *
   * This indicates whether or not the WiFi Display device supports handling of
   * received UIBC cut text events.
   */
  vnc_bool_t cutTextSupported;

} VNCWiFiDisplayUIBCMirrorLinkConfiguration;

/**
 * \brief The UIBC States that a VNC WiFi Display Sink may assume.
 *
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 */
typedef enum
{
  /**
   * \brief Placeholder value for variable initialization.
   */
  VNCWiFiDisplaySinkUIBCStateNone = 0,

  /**
   * \brief The UIBC connection is inactive.
   *
   * SDK APIs and callbacks that utilise the UIBC connection may not
   * be used.
   */
  VNCWiFiDisplaySinkUIBCStateInactive = 1,

  /**
   * \brief The UIBC connection is active.
   *
   * SDK APIs and callbacks that utilise the UIBC connection shall
   * be available for use, assuming their particular configurations
   * have been selected.
   */
  VNCWiFiDisplaySinkUIBCStateActive = 2,

} VNCWiFiDisplaySinkUIBCState;

/**
 * \brief Possible WiFi Display states.
 *
 * \see VNCWiFiDisplaySinkStateChangedCallback
 */
typedef enum
{
  /**
   * \brief Placeholder value for variable initialization.
   */
  VNCWiFiDisplayStateNone = 0,

  /**
   * \brief The WiFi Display session is being established.
   */
  VNCWiFiDisplayStateInit = 1,

  /**
   * \brief The WiFi Display session is established and the RTP media stream
   * is playing.
   */
  VNCWiFiDisplayStatePlaying = 2,

  /**
   * \brief The WiFi Display session is established and the RTP media stream
   * is paused.
   */
  VNCWiFiDisplayStatePaused = 3,

} VNCWiFiDisplayState;

/**
 * \brief A single information entry in a MirrorLink UI context event.
 *
 * \see VNCWiFiDisplaySinkMirrorLinkUIContextEventCallback
 */
typedef struct
{
  /**
   * \brief X-Position of the rectangle covered by this entry.
   */
  vnc_uint16_t xPosition;

  /**
   * \brief Y-Position of the rectangle covered by this entry.
   */
  vnc_uint16_t yPosition;

  /**
   * \brief Width of the rectangle covered by this entry.
   */
  vnc_uint16_t width;

  /**
   * \brief Height of the rectangle covered by this entry.
   */
  vnc_uint16_t height;

  /**
   * \brief The unique identifier of the application that the user interface is
   * originating from.
   *
   * Application identifiers are retrieved from the MirrorLink UPnP session.
   * This may be zero if there is no application.
   */
  vnc_uint32_t applicationUniqueId;

  /**
   * \brief The level of trust that the information in the applicationCategory
   * field is correct.
   */
  VNCTrustLevel applicationCategoryTrustLevel;

  /**
   * \brief The level of trust that the information in the content category
   * field is correct.
   */
  VNCTrustLevel contentCategoryTrustLevel;

  /**
   * \brief The category and sub-category into which the application falls.
   */
  VNCApplicationCategory applicationCategory;

  /**
   * \brief The category of the content that the application is presenting.
   */
  VNCVisualContentCategory contentCategory;

} VNCWiFiDisplayMirrorLinkUIContextInformation;

/**
 * \brief A single information entry in a MirrorLink UI blocking event.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent
 */
typedef struct
{
  /**
   * \brief UI context information index (starting at 1), which is subject to
   * blocking.
   *
   * This must be zero if the UI blocking event is not related to any UI
   * context information.
   */
  vnc_uint8_t index;

  /**
   * \brief A bit-mask describing the reason for this UI blocking information
   * entry.
   *
   * Use the VNCFramebufferBlockReason enumeration to set the bits in this
   * bit-mask.
   *
   * \see VNCFramebufferBlockReason
   */
  vnc_uint16_t blockReason;

} VNCWiFiDisplayMirrorLinkUIBlockingInformation;

/**
 * \brief A single information entry in a MirrorLink audio context event.
 *
 * \see VNCWiFiDisplaySinkMirrorLinkAudioContextEventCallback
 */
typedef struct
{
  /**
   * \brief The unique identifier of the application that the audio is
   * originating from.
   *
   * Application identifiers are retrieved from the MirrorLink UPnP session.
   * This may be zero if there is no application.
   */
  vnc_uint32_t applicationUniqueId;

  /**
   * \brief The category and sub-category into which the application falls.
   */
  VNCApplicationCategory applicationCategory;

} VNCWiFiDisplayMirrorLinkAudioContextInformation;

/**
 * \brief A single information entry in a MirrorLink audio blocking event.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent
 */
typedef struct
{
  /**
   * \brief Audio context information index (starting at 1), which is subject
   * to blocking.
   *
   * This must be zero if the audio blocking event is not related to any audio
   * context information.
   */
  vnc_uint8_t index;

  /**
   * \brief A bit-mask describing the reason for this audio blocking
   * information entry.
   *
   * Use the VNCAudioBlockReason enumeration to set the bits in this bit-mask.
   *
   * \see VNCAudioBlockReason
   */
  vnc_uint16_t blockReason;

} VNCWiFiDisplayMirrorLinkAudioBlockingInformation;

#ifdef __cplusplus
}
#endif

#endif  // UUID_f014b81a_01d2_11e4_98ee_7befe8cb5334
